//=============================================================================
// SkillLearningSystem_v23.js
//=============================================================================

/*:ja
 * @target MZ
 * @plugindesc スキル習得システム（強化説明・最大時非表示・SP獲得メッセON/OFF）
 * @author キンシ
 * @version 2.3.0
 *
 * @help
 * ============================================================================
 * 【概要】
 * SPを消費してスキルを習得・強化。職業制限、強化内容説明（最大レベル時は非表示）、SPリセット機能付き。
 * 戦闘後SP獲得メッセージは表示ON/OFFをプラグインパラメータで切替可能です。
 *
 * 【スキルデータベース設定例】
 * <SkillType:Basic>
 * <SPCost:10>
 * <UpgradeSkillId:101>
 * <UpgradeDesc:威力が50→70に上昇>
 * <RequiredClasses:1,2>
 * <HelpText:炎属性の単体攻撃>
 *
 * 【利用規約】 MITライセンス
 * ============================================================================
 *
 * @param MenuCommandName
 * @text メニューコマンド名
 * @type string @default スキル習得
 * @param SpLabel
 * @text SPラベル
 * @type string @default SP
 * @param SpVariableName
 * @text SP変数名 (内部管理用)
 * @type string @default _skillPoints @optional
 * @param SpGainMessage
 * @text SP獲得メッセージ (戦闘終了時のみ使用)
 * @desc (%1:数値, %2:SPラベル)
 * @type string @default %1 %2 獲得！
 * @param ShowSpGainMessage
 * @text 戦闘後SP獲得メッセージ表示
 * @type boolean
 * @desc 終了時のSP獲得メッセージを表示する（ON）、しない（OFF）
 * @default true
 * @param DefaultSPCost
 * @text デフォルトSPコスト
 * @type number @min 1 @default 100
 * @param AdvancedSkillSwitchId
 * @text アドバンススキル解放スイッチID
 * @type switch @default 1
 * @param LearnSound
 * @text 習得/強化 SE
 * @type file @dir audio/se/ @default Shop1
 * @param BuzzerSound
 * @text 不可 SE
 * @type file @dir audio/se/ @default Buzzer1
 * @param ResetSound
 * @text リセットSE
 * @type file @dir audio/se/ @default Powerup1
 * @command resetSkills
 * @text スキルリセット
 * @desc 指定したアクターのスキルをリセットし、消費SPを返却します。
 * @arg actorId
 * @text 対象アクターID
 * @type actor @default 1
 * @arg confirmMessage
 * @text 確認メッセージ
 * @desc (%1:アクター名, %2:返却SP量, %3:SPラベル) 空欄なら確認なし。
 * @type multiline_string @default %1 のスキルをリセットします。\n消費した %2 %3 が戻ります。\nよろしいですか？
 * @arg completeMessage
 * @text 完了メッセージ
 * @desc (%1:アクター名)
 * @type string @default %1 のスキルをリセットしました。
 */

(() => {
'use strict';

const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)?.[1] || "SkillLearningSystem_v23";
const parameters = PluginManager.parameters(pluginName);

const menuCommandName = parameters['MenuCommandName'] || 'スキル習得';
const spLabel = parameters['SpLabel'] || 'SP';
const spVariableName = parameters['SpVariableName'] || '_skillPoints';
const spGainMessage = parameters['SpGainMessage'] || '%1 %2 獲得！';
const showSpGainMessage = (parameters['ShowSpGainMessage'] ?? 'true') === 'true';
const defaultSPCost = Number(parameters['DefaultSPCost'] || 100);
const advancedSkillSwitchId = Number(parameters['AdvancedSkillSwitchId'] || 1);
const learnSound = { name: parameters['LearnSound'] || "Shop1", volume: 90, pitch: 100, pan: 0 };
const buzzerSound = { name: parameters['BuzzerSound'] || "Buzzer1", volume: 90, pitch: 100, pan: 0 };
const resetSound = { name: parameters['ResetSound'] || "Powerup1", volume: 90, pitch: 100, pan: 0 };

//-------------------------------------
// Game_Actor（主要ロジック・初期スキル・強化・リセット）
//-------------------------------------
const _Game_Actor_initMembers = Game_Actor.prototype.initMembers;
Game_Actor.prototype.initMembers = function() { _Game_Actor_initMembers.call(this); this[spVariableName] = 0; };
Game_Actor.prototype.sp = function() { return this[spVariableName] || 0; };
Game_Actor.prototype.gainSp = function(amount) { const value = this.sp() + amount; this[spVariableName] = Math.max(0, value); };
Game_Actor.prototype.loseSp = function(amount) { this.gainSp(-amount); };
Game_Actor.prototype.getSkillSPCost = function(skill) { if (!skill || !skill.note) return defaultSPCost; const match = skill.note.match(/<SPCost:(\d+)>/i); return match ? parseInt(match[1]) : defaultSPCost; };
Game_Actor.prototype.getSkillType = function(skill) { if (!skill || !skill.note) return null; const match = skill.note.match(/<SkillType:(Basic|Advanced)>/i); return match ? match[1].toLowerCase() : null; };
Game_Actor.prototype.getUpgradeSkillId = function(skill) { if (!skill || !skill.note) return 0; const match = skill.note.match(/<UpgradeSkillId:(\d+)>/i); return match ? parseInt(match[1]) : 0; };
Game_Actor.prototype.getStoryFlag = function(skill) { if (!skill || !skill.note) return 0; const match = skill.note.match(/<StoryFlag:(\d+)>/i); return match ? parseInt(match[1]) : 0; };
Game_Actor.prototype.getSkillHelpText = function(skill) { return (skill && skill.meta && skill.meta.HelpText) ? String(skill.meta.HelpText) : null; };
Game_Actor.prototype.getUpgradeDesc = function(skill) { if (!skill || !skill.note) return ""; const match = skill.note.match(/<UpgradeDesc:(.+?)>/i); return match ? match[1].trim() : ""; };
Game_Actor.prototype.isSkillAvailableForLearning = function(skill) {
    if (!skill) return false;
    const requiredClassesMatch = skill.note.match(/<RequiredClasses:\s*([\d\s,]+)>/i);
    if (requiredClassesMatch) {
        const requiredClassIds = requiredClassesMatch[1].split(',').map(id => parseInt(id.trim())).filter(id => !isNaN(id));
        if (requiredClassIds.length > 0 && !requiredClassIds.includes(this._classId)) return false;
    }
    const skillType = this.getSkillType(skill);
    if (!skillType) return false;
    if (skillType === 'advanced' && !$gameSwitches.value(advancedSkillSwitchId)) return false;
    const storyFlag = this.getStoryFlag(skill);
    if (storyFlag > 0 && !$gameSwitches.value(storyFlag)) return false;
    const skillId = skill.id;
    const hasThisSkill = this.hasSkill(skillId);
    const upgradeId = this.getUpgradeSkillId(skill);
    const isMaxLevel = upgradeId <= 0 || !$dataSkills[upgradeId];
    let currentCheckId = upgradeId;
    while (currentCheckId > 0 && $dataSkills[currentCheckId]) {
        if (this.hasSkill(currentCheckId)) return false;
        const nextUpgrade = this.getUpgradeSkillId($dataSkills[currentCheckId]);
        if (nextUpgrade > 0 && nextUpgrade !== currentCheckId) currentCheckId = nextUpgrade; else break;
    }
    if (hasThisSkill) return isMaxLevel;
    const previousSkill = $dataSkills.find(s => s && this.getUpgradeSkillId(s) === skillId);
    if (previousSkill && !this.hasSkill(previousSkill.id)) return false;
    return true;
};
Game_Actor.prototype.canLearnOrUpgradeSkill = function(skill) {
    if (!skill) return false;
    const upgradeId = this.getUpgradeSkillId(skill);
    const isMaxLevel = upgradeId <= 0 || !$dataSkills[upgradeId];
    if (this.hasSkill(skill.id) && isMaxLevel) return false;
    const cost = this.getSkillSPCost(skill);
    return this.sp() >= cost;
};
Game_Actor.prototype.executeLearnSkill = function(skillId) {
    const skill = $dataSkills[skillId];
    if (!skill || this.hasSkill(skillId) || !this.canLearnOrUpgradeSkill(skill)) return false;
    const cost = this.getSkillSPCost(skill);
    this.loseSp(cost);
    this.learnSkill(skillId);
    const previousSkill = $dataSkills.find(s => s && this.getUpgradeSkillId(s) === skillId);
    if (previousSkill && this.hasSkill(previousSkill.id)) this.forgetSkill(previousSkill.id);
    this.refresh();
    return true;
};
Game_Actor.prototype.getInitialSkills = function() {
    let i = [];
    const actorData = $dataActors[this.actorId()];
    if (actorData && Array.isArray(actorData.skills)) i = i.concat(actorData.skills.map(s => s && s.skillId).filter(id => !!id));
    const c = this.currentClass();
    if (c && Array.isArray(c.learnings)) c.learnings.forEach(l => { if (l && l.level === 1 && l.skillId) i.push(l.skillId); });
    return [...new Set(i)];
};
Game_Actor.prototype.getSkillUpgradeChain = function(skillId) {
    const chain = [];
    let currentId = skillId;
    const checkedIds = new Set();
    while (currentId > 0 && $dataSkills[currentId] && !checkedIds.has(currentId)) {
        chain.push(currentId);
        checkedIds.add(currentId);
        const previousSkill = $dataSkills.find(s => s && this.getUpgradeSkillId(s) === currentId);
        currentId = previousSkill ? previousSkill.id : 0;
    }
    return chain.reverse();
};
// スキルリセット：初期スキルチェインは初期スキルに戻す＆強化分のみSP返却
Game_Actor.prototype.resetSkills = function() {
    let totalSpRefund = 0;
    const initialSkills = this.getInitialSkills();
    const skillsToForget = [];
    const skillsToRelearn = new Set();
    const processedChains = new Set();
    const currentLearnedSkills = [...this._skills];

    for (const learnedSkillId of currentLearnedSkills) {
        const chain = this.getSkillUpgradeChain(learnedSkillId);
        if (!chain || chain.length === 0) continue;
        const chainStartSkillId = chain[0];
        const isInitialChain = initialSkills.includes(chainStartSkillId);

        // SP返却額計算
        if (!processedChains.has(chainStartSkillId)) {
            let chainSpCost = 0;
            for (const chainSkillId of chain) {
                if (isInitialChain && chainSkillId === chainStartSkillId) continue; // 初期スキル分は返却しない
                const skillInChain = $dataSkills[chainSkillId];
                if (skillInChain) chainSpCost += this.getSkillSPCost(skillInChain);
            }
            totalSpRefund += chainSpCost;
            processedChains.add(chainStartSkillId);
        }
        // 忘れるスキル
        skillsToForget.push(learnedSkillId);
        // 初期スキルチェインの場合は初期スキルを再習得
        if (isInitialChain) skillsToRelearn.add(chainStartSkillId);
    }
    for (const skillId of skillsToForget) this.forgetSkill(skillId);
    for (const skillId of skillsToRelearn) if (!this.hasSkill(skillId)) this.learnSkill(skillId);
    this.gainSp(totalSpRefund);
    this.refresh();
    return totalSpRefund;
};
const _Game_Actor_onLoad = Game_Actor.prototype.onLoad;
Game_Actor.prototype.onLoad = function() { _Game_Actor_onLoad.call(this); if (this[spVariableName] === undefined) this[spVariableName] = 0; };

//--- Game_Action ---//
const _Game_Action_apply = Game_Action.prototype.apply;
Game_Action.prototype.apply = function(target) {
    _Game_Action_apply.call(this, target);
    const item = this.item(); const subject = this.subject(); if (!item || !item.note || !target) return;
    const targetSpGainMatch = item.note.match(/<TargetSPGain:(\d+)>/i);
    if (targetSpGainMatch && target.isActor() && !this.isForUser()) { const t = parseInt(targetSpGainMatch[1]); target.gainSp(t); }
    const userSpGainMatch = item.note.match(/<SPGain:(\d+)>/i);
    if (userSpGainMatch && subject.isActor()) { if (DataManager.isItem(item) && !item.consumable) { } else { const u = parseInt(userSpGainMatch[1]); subject.gainSp(u); } }
};

//--- Game_Enemy ---//
Game_Enemy.prototype.sp = function() { const m=this.enemy().note.match(/<SP:(\d+)>/i); return m?parseInt(m[1]):0; };

//--- BattleManager ---//
const _BM_makeR = BattleManager.makeRewards; BattleManager.makeRewards = function() { _BM_makeR.call(this); this._rewards.sp = $gameTroop.deadMembers().reduce((r, e) => r + e.sp(), 0); };
const _BM_gainR = BattleManager.gainRewards; BattleManager.gainRewards = function() { _BM_gainR.call(this); this.gainSp(); };
// ON/OFFでSP獲得メッセージの表示切替
BattleManager.gainSp = function() {
    const s = this._rewards.sp;
    if (s > 0) {
        $gameParty.aliveMembers().forEach(a => a.gainSp(s));
        if (showSpGainMessage) {
            $gameMessage.add(spGainMessage.format(s, spLabel));
        }
    }
};

//--- Window_MenuCommand ---//
const _WMC_addOrg = Window_MenuCommand.prototype.addOriginalCommands;
Window_MenuCommand.prototype.addOriginalCommands = function() { _WMC_addOrg.call(this); this.addCommand(menuCommandName, 'skillLearn', true); };

//--- Scene_Menu ---//
const _SM_createCmd = Scene_Menu.prototype.createCommandWindow;
Scene_Menu.prototype.createCommandWindow = function() { _SM_createCmd.call(this); this._commandWindow.setHandler('skillLearn', this.commandPersonal.bind(this)); };
const _SM_onPersOk = Scene_Menu.prototype.onPersonalOk;
Scene_Menu.prototype.onPersonalOk = function() { if (this._commandWindow.currentSymbol() === 'skillLearn') { SceneManager.push(Scene_SkillLearn); } else { _SM_onPersOk.call(this); } };

//-------------------------------------
// ウィンドウクラス定義
//-------------------------------------
function Window_SkillStatus() { this.initialize(...arguments); }
Window_SkillStatus.prototype = Object.create(Window_StatusBase.prototype); Window_SkillStatus.prototype.constructor = Window_SkillStatus;
Window_SkillStatus.prototype.initialize = function(r){ Window_StatusBase.prototype.initialize.call(this,r); this._actor=null; };
Window_SkillStatus.prototype.setActor = function(a){ if(this._actor!==a){ this._actor=a; this.refresh(); } };
Window_SkillStatus.prototype.refresh = function(){ this.contents.clear(); if(!this._actor) return; const w=this.innerWidth-this.itemPadding()*2; const sx=w*0.70; const sw=w*0.30; this.drawActorName(this._actor,this.itemPadding(),0,w/2); this.drawActorLevel(this._actor,this.itemPadding()+w/2,0); this.changeTextColor(ColorManager.systemColor()); this.drawText(spLabel,sx,0,60); this.resetTextColor(); this.drawText(this._actor.sp(),sx+60,0,sw-60,'right'); };

function Window_SkillLearnList() { this.initialize(...arguments); }
Window_SkillLearnList.prototype = Object.create(Window_Selectable.prototype); Window_SkillLearnList.prototype.constructor = Window_SkillLearnList;
Window_SkillLearnList.prototype.initialize = function(r){ Window_Selectable.prototype.initialize.call(this,r); this._actor=null; this._data=[]; };
Window_SkillLearnList.prototype.setActor = function(a){ if(this._actor!==a){ this._actor=a; this.refresh(); this.scrollTo(0,0); } };
Window_SkillLearnList.prototype.maxCols = function(){ return 1; };
Window_SkillLearnList.prototype.maxItems = function(){ return this._data?this._data.length:0; };
Window_SkillLearnList.prototype.item = function(){ return this.index()>=0?this._data[this.index()]:null; };
Window_SkillLearnList.prototype.isCurrentItemEnabled = function(){ const s=this.item(); if(!this._actor || !s) return false; return this._actor.canLearnOrUpgradeSkill(s); };
Window_SkillLearnList.prototype.makeItemList = function(){ this._data=[]; if(this._actor){ this._data=$dataSkills.filter(s=>s && this._actor.isSkillAvailableForLearning(s)); } };
Window_SkillLearnList.prototype.refresh = function(){ this.makeItemList(); this.createContents(); this.drawAllItems(); };
Window_SkillLearnList.prototype.drawItem = function(index) {
    const skill = this._data[index];
    const rect = this.itemLineRect(index);
    if (!this._actor || !skill) return;
    const costWidth = 80; const statusWidth = 90; const nameWidth = rect.width - costWidth - statusWidth - this.itemPadding() * 2;
    const upgradeId = this._actor.getUpgradeSkillId(skill);
    const isMaxLevel = upgradeId <= 0 || !$dataSkills[upgradeId];
    const hasLearned = this._actor.hasSkill(skill.id);
    const canExecute = this._actor.canLearnOrUpgradeSkill(skill);
    const enabled = canExecute || !(hasLearned && isMaxLevel); this.changePaintOpacity(enabled);
    this.drawItemName(skill, rect.x, rect.y, nameWidth);
    let statusText = ""; if (hasLearned && isMaxLevel) { statusText = "最大"; this.changeTextColor(ColorManager.textColor(8)); } else { statusText = "習得"; this.changeTextColor(canExecute ? ColorManager.normalColor() : ColorManager.textColor(8)); }
    this.drawText(statusText, rect.x + nameWidth, rect.y, statusWidth, 'right'); this.resetTextColor();
    const cost = this._actor.getSkillSPCost(skill); const costText = (hasLearned && isMaxLevel) ? "－" : `${cost}${spLabel}`;
    this.changeTextColor(ColorManager.normalColor()); this.drawText(costText, rect.x + nameWidth + statusWidth, rect.y, costWidth, 'right'); this.resetTextColor();
    this.changePaintOpacity(true);
};
Window_SkillLearnList.prototype.updateHelp = function() {
    const skill = this.item();
    let helpText = "";
    if (skill && this._actor) {
        const customHelp = this._actor.getSkillHelpText(skill);
        helpText = customHelp ? customHelp : (skill.description || "");
    }
    this._helpWindow.setText(helpText);
};

// スキル詳細ウィンドウ（強化内容説明・最大スキル習得時非表示）
function Window_SkillLearnDetail() { this.initialize(...arguments); }
Window_SkillLearnDetail.prototype = Object.create(Window_Base.prototype); Window_SkillLearnDetail.prototype.constructor = Window_SkillLearnDetail;
Window_SkillLearnDetail.prototype.initialize = function(rect){ Window_Base.prototype.initialize.call(this,rect); this._skill=null; this._actor=null; };
Window_SkillLearnDetail.prototype.setActor = function(a){ if(this._actor!==a){ this._actor=a; this.refresh(); } };
Window_SkillLearnDetail.prototype.setSkill = function(s){ if(this._skill!==s){ this._skill=s; this.refresh(); } };
Window_SkillLearnDetail.prototype.refresh = function() {
    this.contents.clear();
    if (!this._skill || !this._actor) return;
    const skill = this._skill;
    const actor = this._actor;
    const lineHeight = this.lineHeight();
    const width = this.innerWidth - this.itemPadding() * 2;
    let y = 0;
    // 名前
    this.drawItemName(skill, this.itemPadding(), y, width * 0.7);
    y += lineHeight;
    // 説明文
    let desc = skill.description || "";
    if (desc) {
        this.drawTextEx(desc, this.itemPadding(), y);
        y += this.textSizeEx(desc).height;
    }
    // 強化内容説明（最大スキル習得時は非表示）
    let maxSkillId = skill.id;
    let upgradeId = actor.getUpgradeSkillId(skill);
    while (upgradeId > 0 && $dataSkills[upgradeId]) {
        maxSkillId = upgradeId;
        upgradeId = actor.getUpgradeSkillId($dataSkills[upgradeId]);
    }
    const isMaxSkillLearned = actor.hasSkill(maxSkillId);
    const upgradeDesc = actor.getUpgradeDesc(skill);
    if (!isMaxSkillLearned && upgradeDesc) {
        y += 4;
        this.changeTextColor(ColorManager.systemColor());
        this.drawText("強化内容:", this.itemPadding(), y, width);
        this.resetTextColor();
        y += lineHeight;
        this.drawTextEx(upgradeDesc, this.itemPadding(), y);
    }
};

function Scene_SkillLearn() { this.initialize(...arguments); }
Scene_SkillLearn.prototype = Object.create(Scene_MenuBase.prototype); Scene_SkillLearn.prototype.constructor = Scene_SkillLearn;
Scene_SkillLearn.prototype.initialize = function() { Scene_MenuBase.prototype.initialize.call(this); };
Scene_SkillLearn.prototype.create = function() {
    Scene_MenuBase.prototype.create.call(this);
    this.createHelpWindow();
    this.createSkillStatusWindow();
    this.createSkillListWindow();
    this.createSkillDetailWindow();
};
Scene_SkillLearn.prototype.start = function() {
    Scene_MenuBase.prototype.start.call(this);
    this.refreshActor();
    this._skillListWindow.activate();
    this._skillListWindow.select(0);
};
Scene_SkillLearn.prototype.refreshActor = function() {
    const actor = this.actor();
    this._skillStatusWindow.setActor(actor);
    if (this._skillListWindow) this._skillListWindow.setActor(actor);
    if (this._skillDetailWindow) this._skillDetailWindow.setActor(actor);
    if (this._skillDetailWindow) this._skillDetailWindow.setSkill(null);
    if (this._skillListWindow) this._skillListWindow.select(-1);
};
Scene_SkillLearn.prototype.createHelpWindow = function() {
    const rect = this.helpWindowRect();
    this._helpWindow = new Window_Help(rect);
    this.addWindow(this._helpWindow);
};
Scene_SkillLearn.prototype.createSkillStatusWindow = function() {
    const rect = this.skillStatusWindowRect();
    this._skillStatusWindow = new Window_SkillStatus(rect);
    this.addWindow(this._skillStatusWindow);
};
Scene_SkillLearn.prototype.skillStatusWindowRect = function() {
    const ww = Graphics.boxWidth;
    const wh = this.calcWindowHeight(1, false) + 8;
    const wx = 0;
    const wy = this.mainAreaTop();
    return new Rectangle(wx, wy, ww, wh);
};
Scene_SkillLearn.prototype.createSkillListWindow = function() {
    const rect = this.skillListWindowRect();
    this._skillListWindow = new Window_SkillLearnList(rect);
    this._skillListWindow.setHelpWindow(this._helpWindow);
    this._skillListWindow.setHandler('ok', this.onSkillOk.bind(this));
    this._skillListWindow.setHandler('cancel', this.onSkillCancel.bind(this));
    this.addWindow(this._skillListWindow);
};
Scene_SkillLearn.prototype.skillListWindowRect = function() {
    const statusRect = this.skillStatusWindowRect();
    const detailRect = this.skillDetailWindowRect();
    const wx = 0;
    const wy = statusRect.y + statusRect.height;
    const ww = Graphics.boxWidth;
    const wh = detailRect.y - wy;
    return new Rectangle(wx, wy, ww, wh);
};
Scene_SkillLearn.prototype.createSkillDetailWindow = function() {
    const rect = this.skillDetailWindowRect();
    this._skillDetailWindow = new Window_SkillLearnDetail(rect);
    this.addWindow(this._skillDetailWindow);
    if (this._skillListWindow) {
        const list = this._skillListWindow;
        list.setHandler('ok', this.onSkillOk.bind(this));
        list.setHandler('cancel', this.onSkillCancel.bind(this));
        // スキル選択時に詳細ウィンドウに反映
        const _updateHelp = list.updateHelp;
        list.updateHelp = function() {
            _updateHelp.call(this);
            if (this._detailWindow) this._detailWindow.setSkill(this.item());
        }.bind(list);
        list._detailWindow = this._skillDetailWindow;
    }
};
Scene_SkillLearn.prototype.skillDetailWindowRect = function() {
    const ww = Graphics.boxWidth;
    const wh = this.calcWindowHeight(4, true);
    const wx = 0;
    const wy = this.mainAreaBottom() - wh;
    return new Rectangle(wx, wy, ww, wh);
};
Scene_SkillLearn.prototype.onActorChange = function() {
    Scene_MenuBase.prototype.onActorChange.call(this);
    this.refreshActor();
    this._skillListWindow.activate();
    this._skillListWindow.select(0);
};
Scene_SkillLearn.prototype.onSkillCancel = function() {
    this._skillListWindow.select(-1);
    this._skillListWindow.deactivate();
    this._helpWindow.clear();
    this.popScene();
};
Scene_SkillLearn.prototype.onSkillOk = function() {
    const skill = this._skillListWindow.item();
    const actor = this.actor();
    if (!skill || !actor) {
        this.playBuzzerSound();
        this._skillListWindow.activate();
        return;
    }
    if (actor.executeLearnSkill(skill.id)) {
        this.playLearnSound();
        this._skillStatusWindow.refresh();
        this._skillListWindow.refresh();
        this._skillListWindow.activate();
    } else {
        this.playBuzzerSound();
        this._skillListWindow.activate();
    }
};
Scene_SkillLearn.prototype.playLearnSound = function() { AudioManager.playSe(learnSound); };
Scene_SkillLearn.prototype.playBuzzerSound = function() { AudioManager.playSe(buzzerSound); };

//------------------
// プラグインコマンド
//------------------
PluginManager.registerCommand(pluginName, "resetSkills", function(args) {
    const actorId=Number(args.actorId);
    const actor=$gameActors.actor(actorId);
    if(!actor){ console.error(`[${pluginName}] ResetSkills: Actor ID ${actorId} not found.`); return; }
    const cMsg=args.confirmMessage;
    const coMsg=args.completeMessage||"%1 のスキルをリセットしました。";
    let r=0;
    const iS=actor.getInitialSkills();
    const pChains = new Set();
    for (const id of actor._skills) {
        const chain = actor.getSkillUpgradeChain(id);
        if (!chain || chain.length === 0) continue;
        const chainStartSkillId = chain[0];
        if (pChains.has(chainStartSkillId)) continue;
        let chainSpCost = 0;
        let isInitialChain = iS.includes(chainStartSkillId);
        for (const cId of chain) {
            if(isInitialChain && cId === chainStartSkillId) continue;
            const sInC = $dataSkills[cId];
            if (sInC) { chainSpCost += actor.getSkillSPCost(sInC); }
        }
        r += chainSpCost;
        pChains.add(chainStartSkillId);
    }
    const pR=()=>{ const rs=actor.resetSkills(); AudioManager.playSe(resetSound); $gameMessage.add(coMsg.format(actor.name())); };
    if(cMsg){
        const m=cMsg.format(actor.name(),r,spLabel);
        $gameMessage.setChoices(["はい","いいえ"],1,0);
        $gameMessage.setChoiceCallback(c=>{ if(c===0){ pR(); } });
        $gameMessage.add(m+"\\^");
        this.setWaitMode("message");
    } else { pR(); }
});

})();
